<?php

namespace Modules\Notification\Models;

use CodeIgniter\Model;

class NotificationModel extends Model
{
    protected $DBGroup              = 'default';
    protected $table                = 'notifications';
    protected $primaryKey           = 'id';
    protected $useAutoIncrement     = true;
    protected $insertID             = 0;
    protected $returnType           = 'object';
    protected $useSoftDeletes       = true;
    protected $protectFields        = true;
    protected $allowedFields        = [
        'id', 'subject', 'message', 'notification_type', 'target_group', 
        'trip_id', 'notification_date', 'repeat_option', 'status', 
        'is_push_notification', 'created_by'
    ];

    // Dates
    protected $useTimestamps        = true;
    protected $dateFormat           = 'datetime';
    protected $createdField         = 'created_at';
    protected $updatedField         = 'updated_at';
    protected $deletedField         = 'deleted_at';

    // Validation
    protected $validationRules      = [
        'subject' => 'required|min_length[3]|max_length[255]',
        'message' => 'required|min_length[10]',
        'notification_type' => 'required|in_list[manual,automatic]',
        'target_group' => 'required|in_list[all_users,specific_trip]',
        'notification_date' => 'required|valid_date',
        'repeat_option' => 'in_list[none,daily,weekly,monthly]',
        'status' => 'in_list[draft,scheduled,sent,cancelled]',
        'created_by' => 'required|integer'
    ];
    
    protected $validationMessages   = [
        'subject' => [
            'required' => 'Subject is required',
            'min_length' => 'Subject must be at least 3 characters long',
            'max_length' => 'Subject cannot exceed 255 characters'
        ],
        'message' => [
            'required' => 'Message is required',
            'min_length' => 'Message must be at least 10 characters long'
        ],
        'notification_type' => [
            'required' => 'Notification type is required',
            'in_list' => 'Invalid notification type'
        ],
        'target_group' => [
            'required' => 'Target group is required',
            'in_list' => 'Invalid target group'
        ],
        'notification_date' => [
            'required' => 'Notification date is required',
            'valid_date' => 'Please provide a valid date'
        ]
    ];
    
    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks       = true;
    protected $beforeInsert         = [];
    protected $afterInsert          = [];
    protected $beforeUpdate         = [];
    protected $afterUpdate          = [];
    protected $beforeFind           = [];
    protected $afterFind            = [];
    protected $beforeDelete         = [];
    protected $afterDelete          = [];

    /**
     * Get notifications by status
     */
    public function getByStatus($status)
    {
        return $this->where('status', $status)->findAll();
    }

    /**
     * Get scheduled notifications that are ready to be sent
     */
    public function getScheduledForSending()
    {
        return $this->where('status', 'scheduled')
                    ->where('notification_date <=', date('Y-m-d H:i:s'))
                    ->findAll();
    }

    /**
     * Get notifications with trip details
     */
    public function getWithTripDetails()
    {
        return $this->select('notifications.*, trips.trip_name, trips.departure_time, trips.departure_date')
                    ->join('trips', 'trips.id = notifications.trip_id', 'left')
                    ->findAll();
    }

    /**
     * Get notification statistics
     */
    public function getStatistics()
    {
        $db = \Config\Database::connect();
        
        $stats = [];
        $stats['total'] = $this->countAll();
        $stats['sent'] = $this->where('status', 'sent')->countAllResults(false);
        $stats['scheduled'] = $this->where('status', 'scheduled')->countAllResults(false);
        $stats['cancelled'] = $this->where('status', 'cancelled')->countAllResults(false);
        $stats['draft'] = $this->where('status', 'draft')->countAllResults(false);
        
        return $stats;
    }
}